<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Blog;
use App\Models\Cate;
use App\Models\Category;
use App\Models\Certificate;
use App\Models\ContactMessage;
use App\Models\User;
use App\Models\Video;
use App\Models\WebAbout;
use App\Models\WebContact;
use App\Models\WebDonation;
use App\Models\WebHome;
use App\Models\WebProject;
use App\Models\Websettings;
use App\Models\Donation;
use Dom\Document;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Crypt;
use Carbon\Carbon;
use Barryvdh\DomPDF\Facade\Pdf;


use Illuminate\Http\Request;
use LDAP\Result;
use Illuminate\Support\Facades\Mail;
class BlogController extends Controller
{
    // Category start here
    public function category()
    {
        return view('admin.category');
    }
    public function categoryCreate()
    {
        return view('admin.category-create');
    }
    public function categoryStore(Request $request)
    {
        $request->validate([
            'name' => 'required',
            'slug' => 'nullable|string|max:255',
            'description' => 'required|string',
            'status' => 'required|string',
        ]);
        $slug = $request->slug ?: Str::slug($request->title);
        $slugCount = Category::where('slug', $slug)->count();
        if ($slugCount > 0) {
            $slug .= '-' . time();
        }
        Category::create([
            'name' => $request->name,
            'slug' => $slug,
            'description' => $request->description,
            'status' => $request->status
        ]);
        return redirect()->route('category')->with('status', 'Category created successfully.');
    }
    public function getCategoryData()
    {
        $categorys = Category::orderBy('id', 'desc')->get();
        $result = [];
        $i = 1;

        foreach ($categorys as $category) {
            // Create status dropdown
            $statusOptions = '
                <select class="form-control category-status" data-id="' . $category->id . '">
                    <option value="Active"' . ($category->status === 'Active' ? ' selected' : '') . '>Active</option>
                    <option value="Inactive"' . ($category->status === 'Inactive' ? ' selected' : '') . '>Inactive</option>
                </select>';

            $editBtn = "<a class='btn btn-info editBtn' data-id='{$category->id}'>Edit</a>";
            $deleteBtn = "<a class='btn btn-danger deleteBtn' data-id='{$category->id}'>Delete</a>";

            $result['data'][] = [
                $i++,
                $category->name,
                $statusOptions,
                $editBtn . ' ' . $deleteBtn
            ];
        }

        return response()->json($result);
    }
    public function categoryEdit($id)
    {
        $category = Category::findOrFail($id);
        return view('admin.category-edit', compact('category'));
    }

    public function categoryUpdate(Request $request, $id)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'status' => 'required|in:Active,Inactive',
        ]);

        $category = Category::findOrFail($id);
        $category->name = $request->name;
        $category->slug = \Str::slug($request->name); // Auto-generate slug
        $category->description = $request->description;
        $category->status = $request->status;
        $category->save();

        return redirect()->route('category')->with('status', 'Category updated successfully.');
    }
    public function categoryUpdateStatus(Request $request)
    {
        $category = Category::find($request->id);

        if (!$category) {
            return response()->json(['message' => 'Category not found'], 404);
        }

        $category->status = $request->status;
        $category->save();

        return response()->json(['message' => 'Status updated successfully']);
    }
    public function categoryDelete($id)
    {
        $category = Category::findOrFail($id);
        $category->delete();

        return response()->json(['status' => 'success', 'message' => 'Category deleted successfully.']);
    }


    // Category End here


    // Blog Start  here

    public function blogs()
    {
        return view('admin.blogs');
    }
    public function getBlogsData()
    {
        $blogs = Blog::orderBy('id', 'desc')->get(); // For MongoDB, use id instead of id
        $result = [];
        $i = 1;

        foreach ($blogs as $blog) {

            // Create status dropdown
            $statusOptions = '
            <select class="form-control blog-status" data-id="' . $blog->id . '">
                <option value="Active" ' . ($blog->is_feature === 'Active' ? 'selected' : '') . '>Active</option>
                <option value="Inactive" ' . ($blog->is_feature === 'Inactive' ? 'selected' : '') . '>Inactive</option>
            </select>';

            // Action buttons
            $editBtn = "<a class='btn btn-info editBtn' data-id='{$blog->id}'>Edit</a>";
            $deleteBtn = "<a class='btn btn-danger deleteBtn' data-id='{$blog->id}'>Delete</a>";

            $result['data'][] = [
                $i++,
                $blog->title,
                $blog->slug,
                $statusOptions,
                $editBtn . ' ' . $deleteBtn
            ];
        }

        return response()->json($result);
    }

    public function index()
    {
        $users = User::all();
        $categories = Category::orderBy('id', 'desc')->get();
        return view('admin.blog-create', compact('users', 'categories'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'slug' => 'nullable|string|max:255',
            'category_id' => 'required|string',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif,svg|max:2048',
            'description' => 'required|string',
            'author' => 'required|string|max:255',
            'tag' => 'nullable|string',
            'schedule_date' => 'nullable|date',
            'is_feature' => 'required|string',
        ]);

        // Save image if provided
        $featureImageName = null;
        if ($request->hasFile('image')) {
            $image = $request->file('image');
            $featureImageName = time() . '.' . $image->getClientOriginalExtension();
            $image->move(public_path('uploads/blog'), $featureImageName);
        }


        // Handle slug
        $slug = $request->slug ?: Str::slug($request->title);
        if (Blog::where('slug', $slug)->exists()) {
            $slug .= '-' . time();
        }

        // Parse and save base64 images in summernote HTML content
        $description = $request->description;
        $dom = new \DOMDocument();
        libxml_use_internal_errors(true);
        $dom->loadHTML('<?xml encoding="utf-8" ?>' . $description, LIBXML_HTML_NOIMPLIED | LIBXML_HTML_NODEFDTD);
        libxml_clear_errors();

        $images = $dom->getElementsByTagName('img');
        foreach ($images as $key => $img) {
            $src = $img->getAttribute('src');
            if (preg_match('/^data:image\/(\w+);base64,/', $src, $type)) {
                $data = substr($src, strpos($src, ',') + 1);
                $data = base64_decode($data);
                $imageType = strtolower($type[1]);
                $imageName = time() . $key . '.' . $imageType;
                $imagePath = public_path('uploads/description/' . $imageName);
                file_put_contents($imagePath, $data);
                $img->setAttribute('src', asset('uploads/description/' . $imageName));
            }
        }

        $finalDescription = $dom->saveHTML();

        // Now save the blog (Laravel will encrypt 'description' automatically)
        Blog::create([
            'title' => $request->title,
            'slug' => $slug,
            'category_id' => $request->category_id,
            'image' => $featureImageName,
            'description' => $finalDescription, // encryption handled by $casts
            'author' => $request->author,
            'tag' => $request->tag,
            'schedule_date' => $request->schedule_date,
            'is_feature' => $request->is_feature,
        ]);

        return redirect()->route('blog')->with('status', 'Blog created successfully.');
    }
    public function blogEdit($id)
    {
        $blog = Blog::findOrFail($id);

        $categories = Category::all();
        $users = User::all();

        return view('admin.blog-edit', compact('blog', 'categories', 'users'));
    }
    public function blogUpdate(Request $request, $id)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'category_id' => 'required|string',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif,svg|max:2048',
            'description' => 'required|string',
            'author' => 'required|string|max:255',
            'tag' => 'nullable|string',
            'schedule_date' => 'nullable|date',
            'is_feature' => 'required|string',
        ]);

        $blog = Blog::findOrFail($id);

        // === Handle new feature image upload ===
        if ($request->hasFile('image')) {
            if ($blog->image && file_exists(public_path('uploads/blog/' . $blog->image))) {
                unlink(public_path('uploads/blog/' . $blog->image));
            }

            $image = $request->file('image');
            $imageName = 'feature_' . time() . '.' . $image->getClientOriginalExtension();
            $image->move(public_path('uploads/blog'), $imageName);
            $blog->image = $imageName;
        }

        // === Extract old image filenames from previous description ===
        $oldImagePaths = [];
        $oldDom = new \DOMDocument();
        libxml_use_internal_errors(true);
        $oldDom->loadHTML('<?xml encoding="utf-8" ?>' . $blog->description, LIBXML_HTML_NOIMPLIED | LIBXML_HTML_NODEFDTD);
        libxml_clear_errors();

        foreach ($oldDom->getElementsByTagName('img') as $img) {
            $src = $img->getAttribute('src');
            if (str_contains($src, asset('uploads/description/'))) {
                $filename = basename(parse_url($src, PHP_URL_PATH));
                $oldImagePaths[] = $filename;
            }
        }

        // === Process new description, store new base64 images ===
        $newImagePaths = [];
        $description = $request->description;
        $dom = new \DOMDocument();
        libxml_use_internal_errors(true);
        $dom->loadHTML('<?xml encoding="utf-8" ?>' . $description, LIBXML_HTML_NOIMPLIED | LIBXML_HTML_NODEFDTD);
        libxml_clear_errors();

        foreach ($dom->getElementsByTagName('img') as $key => $img) {
            $src = $img->getAttribute('src');

            if (preg_match('/^data:image\/(\w+);base64,/', $src, $type)) {
                $data = substr($src, strpos($src, ',') + 1);
                $data = base64_decode($data);
                $imageType = strtolower($type[1]);
                $imageName = 'desc_' . time() . $key . '.' . $imageType;
                $imagePath = public_path('uploads/description/' . $imageName);
                file_put_contents($imagePath, $data);
                $img->setAttribute('src', asset('uploads/description/' . $imageName));
                $newImagePaths[] = $imageName;
            } else {
                $filename = basename(parse_url($src, PHP_URL_PATH));
                $newImagePaths[] = $filename;
            }
        }

        $finalDescription = $dom->saveHTML();

        // === Delete removed images from previous description ===
        $deletedImages = array_diff($oldImagePaths, $newImagePaths);

        foreach ($deletedImages as $fileToDelete) {
            $filePath = public_path('uploads/description/' . $fileToDelete);
            if (file_exists($filePath)) {
                unlink($filePath);
            }
        }

        // === Update blog fields ===
        $slug = $request->slug ?: Str::slug($request->title);
        if (Blog::where('slug', $slug)->where('id', '!=', $id)->exists()) {
            $slug .= '-' . time();
        }

        $blog->title = $request->title;
        $blog->slug = $slug;
        $blog->category_id = $request->category_id;
        $blog->description = $finalDescription;
        $blog->author = $request->author;
        $blog->tag = $request->tag;
        $blog->schedule_date = $request->schedule_date;
        $blog->is_feature = $request->is_feature;
        $blog->save();

        return redirect()->route('blog')->with('status', 'Blog updated successfully.');
    }



    public function blogdestroy($id)
    {
        $blog = Blog::findOrFail($id);

        // Delete feature image if it exists
        if ($blog->image && file_exists(public_path('storage/' . $blog->image))) {
            unlink(public_path('storage/' . $blog->image));
        }

        // Delete images from the description
        $dom = new \DOMDocument();
        libxml_use_internal_errors(true);
        $dom->loadHTML($blog->description, LIBXML_HTML_NOIMPLIED | LIBXML_HTML_NODEFDTD);
        libxml_clear_errors();

        $images = $dom->getElementsByTagName('img');
        foreach ($images as $img) {
            $src = $img->getAttribute('src');
            $path = str_replace(asset('/'), '', $src); // Convert full URL to relative path
            $fullPath = public_path($path);
            if (file_exists($fullPath)) {
                unlink($fullPath);
            }
        }

        // Finally, delete the blog entry from the database
        $blog->delete();

        return response()->json(['message' => 'Blog and associated images deleted successfully.']);
    }

    public function updateFeatureStatus(Request $request)
    {
        $blog = Blog::find($request->id);
        if (!$blog) {
            return response()->json([
                'error' => true,
                'message' => 'Blog not found!'
            ]);
        }

        $blog->is_feature = $request->is_feature;
        $blog->save();

        return response()->json([
            'error' => false,
            'message' => 'Feature status updated successfully!'
        ]);
    }

    //Blog section end here 


    //Web Image Start
    public function webimage()
    {

        return view('admin.web-image');
    }
    public function webhome()
    {

        return view('admin.web-home');
    }
    //home page image store
    public function homestore(Request $request)
    {
        $data = [];
        $uploadPath = public_path('uploads/home');

        if (!file_exists($uploadPath)) {
            mkdir($uploadPath, 0755, true);
        }

        // Banner Image
        if ($request->hasFile('banner_image')) {
            $file = $request->file('banner_image');
            $filename = 'banner_' . time() . '.' . $file->getClientOriginalExtension();
            $file->move($uploadPath, $filename);
            $data['banner_image'] = $filename;
        }

        // Campaign Images
        for ($i = 1; $i <= 4; $i++) {
            $inputName = "campaign_image{$i}";
            if ($request->hasFile($inputName)) {
                $file = $request->file($inputName);
                $filename = "{$inputName}_" . time() . '.' . $file->getClientOriginalExtension();
                $file->move($uploadPath, $filename);
                $data[$inputName] = $filename;
            }
        }
        //about images
        for ($i = 1; $i <= 3; $i++) {
            $inputName = "about_image{$i}";
            if ($request->hasFile($inputName)) {
                $file = $request->file($inputName);
                $filename = "{$inputName}_" . time() . '.' . $file->getClientOriginalExtension();
                $file->move($uploadPath, $filename);
                $data[$inputName] = $filename;
            }
        }

        // Community Impact
        if ($request->hasFile('community_impact_image')) {
            $file = $request->file('community_impact_image');
            $filename = 'community_impact_' . time() . '.' . $file->getClientOriginalExtension();
            $file->move($uploadPath, $filename);
            $data['community_impact_image'] = $filename;
        }

        // Volunteer Images
        for ($i = 1; $i <= 2; $i++) {
            $inputName = "volunteer_image{$i}";
            if ($request->hasFile($inputName)) {
                $file = $request->file($inputName);
                $filename = "{$inputName}_" . time() . '.' . $file->getClientOriginalExtension();
                $file->move($uploadPath, $filename);
                $data[$inputName] = $filename;
            }
        }

        $data['section'] = 'home';

        WebHome::create($data);

        return redirect()->route('webimage')->with('status', 'Home Images uploaded successfully!');
    }
    public function webhomeEdit()
    {
        $webhome = WebHome::first();
        return view('admin.web-home-edit', compact('webhome'));
    }
    public function webhomeUpdate(Request $request)
    {
        $webhome = WebHome::first();

        if (!$webhome) {
            return redirect()->back()->with('error', 'No home image data found.');
        }

        $data = [];
        $uploadPath = public_path('uploads/home');

        if (!file_exists($uploadPath)) {
            mkdir($uploadPath, 0755, true);
        }

        // Banner Image
        if ($request->hasFile('banner_image')) {
            if (!empty($webhome->banner_image) && file_exists($uploadPath . '/' . $webhome->banner_image)) {
                unlink($uploadPath . '/' . $webhome->banner_image);
            }
            $file = $request->file('banner_image');
            $filename = 'banner_' . time() . '.' . $file->getClientOriginalExtension();
            $file->move($uploadPath, $filename);
            $data['banner_image'] = $filename;
        }
        //about 
        for ($i = 1; $i <= 3; $i++) {
            $inputName = "about_image{$i}";
            if ($request->hasFile($inputName)) {
                if (!empty($webhome->$inputName) && file_exists($uploadPath . '/' . $webhome->$inputName)) {
                    unlink($uploadPath . '/' . $webhome->$inputName);
                }
                $file = $request->file($inputName);
                $filename = "{$inputName}_" . time() . '.' . $file->getClientOriginalExtension();
                $file->move($uploadPath, $filename);
                $data[$inputName] = $filename;
            }
        }

        // Campaign Images (1 to 4)
        for ($i = 1; $i <= 4; $i++) {
            $inputName = "campaign_image{$i}";
            if ($request->hasFile($inputName)) {
                // Delete old image if it exists
                if (!empty($webhome->$inputName) && file_exists($uploadPath . '/' . $webhome->$inputName)) {
                    unlink($uploadPath . '/' . $webhome->$inputName);
                }

                $file = $request->file($inputName);
                $filename = "{$inputName}_" . time() . '.' . $file->getClientOriginalExtension();
                $file->move($uploadPath, $filename);
                $data[$inputName] = $filename;
            }
        }

        // Community Impact Image
        if ($request->hasFile('community_impact_image')) {
            if (!empty($webhome->community_impact_image) && file_exists($uploadPath . '/' . $webhome->community_impact_image)) {
                unlink($uploadPath . '/' . $webhome->community_impact_image);
            }
            $file = $request->file('community_impact_image');
            $filename = 'community_impact_' . time() . '.' . $file->getClientOriginalExtension();
            $file->move($uploadPath, $filename);
            $data['community_impact_image'] = $filename;
        }

        // Volunteer Images (1 and 2)
        for ($i = 1; $i <= 2; $i++) {
            $inputName = "volunteer_image{$i}";
            if ($request->hasFile($inputName)) {
                // Delete old image if it exists
                if (!empty($webhome->$inputName) && file_exists($uploadPath . '/' . $webhome->$inputName)) {
                    unlink($uploadPath . '/' . $webhome->$inputName);
                }

                $file = $request->file($inputName);
                $filename = "{$inputName}_" . time() . '.' . $file->getClientOriginalExtension();
                $file->move($uploadPath, $filename);
                $data[$inputName] = $filename;
            }
        }

        $webhome->update($data);

        return redirect()->route('webimage')->with('status', 'Home Images updated successfully!');
    }
    //web about image
    public function webabout()
    {
        return view('admin.web-about');
    }
    public function webaboutstore(Request $request)
    {
        $data = [];
        $uploadPath = public_path('uploads/about');

        if (!file_exists($uploadPath)) {
            mkdir($uploadPath, 0755, true);
        }

        // Market Maestro Image
        if ($request->hasFile('market_maestro_image')) {
            $file = $request->file('market_maestro_image');
            $filename = 'market_maestro_' . time() . '.' . $file->getClientOriginalExtension();
            $file->move($uploadPath, $filename);
            $data['market_maestro_image'] = $filename;
        }

        // Foundation Images
        for ($i = 1; $i <= 3; $i++) {
            $inputName = "foundation_image{$i}";
            if ($request->hasFile($inputName)) {
                $file = $request->file($inputName);
                $filename = "{$inputName}_" . time() . '.' . $file->getClientOriginalExtension();
                $file->move($uploadPath, $filename);
                $data[$inputName] = $filename;
            }
        }

        // Founder Images
        for ($i = 1; $i <= 2; $i++) {
            $inputName = "founder_image{$i}";
            if ($request->hasFile($inputName)) {
                $file = $request->file($inputName);
                $filename = "{$inputName}_" . time() . '.' . $file->getClientOriginalExtension();
                $file->move($uploadPath, $filename);
                $data[$inputName] = $filename;
            }
        }

        WebAbout::create($data);

        return redirect()->route('webimage')->with('status', 'About Images uploaded successfully!');
    }
    public function webaboutedit()
    {
        $data = WebAbout::first(); // Ignores $id
        return view('admin.web-about-edit', compact('data'));
    }
    public function webaboutupdate(Request $request)
    {
        $webAbout = WebAbout::first();
        $uploadPath = public_path('uploads/about');

        if (!file_exists($uploadPath)) {
            mkdir($uploadPath, 0755, true);
        }

        $data = [];

        if ($request->hasFile('market_maestro_image')) {
            if (!empty($webAbout->market_maestro_image)) {
                $oldPath = public_path('uploads/about/' . $webAbout->market_maestro_image);
                if (file_exists($oldPath)) {
                    unlink($oldPath);
                }
            }
            $file = $request->file('market_maestro_image');
            $filename = 'market_maestro_' . time() . '.' . $file->getClientOriginalExtension();
            $file->move($uploadPath, $filename);
            $data['market_maestro_image'] = $filename;
        }

        // Foundation Images
        for ($i = 1; $i <= 3; $i++) {
            $inputName = "foundation_image{$i}";
            if ($request->hasFile($inputName)) {
                if (!empty($webAbout->$inputName)) {
                    $oldPath = public_path('uploads/about/' . $webAbout->$inputName);
                    if (file_exists($oldPath)) {
                        unlink($oldPath);
                    }
                }
                $file = $request->file($inputName);
                $filename = "{$inputName}_" . time() . '.' . $file->getClientOriginalExtension();
                $file->move($uploadPath, $filename);
                $data[$inputName] = $filename;
            }
        }

        // Founder Images
        for ($i = 1; $i <= 2; $i++) {
            $inputName = "founder_image{$i}";
            if ($request->hasFile($inputName)) {
                if (!empty($webAbout->$inputName)) {
                    $oldPath = public_path('uploads/about/' . $webAbout->$inputName);
                    if (file_exists($oldPath)) {
                        unlink($oldPath);
                    }
                }
                $file = $request->file($inputName);
                $filename = "{$inputName}_" . time() . '.' . $file->getClientOriginalExtension();
                $file->move($uploadPath, $filename);
                $data[$inputName] = $filename;
            }
        }

        $webAbout->update($data);

        return redirect()->route('webimage')->with('status', 'About Images updated successfully!');
    }
    public function webproject()
    {
        return view('admin.web-project');
    }
    public function webprojectstore(Request $request)
    {
        $data = [];
        $uploadPath = public_path('uploads/project');

        if (!file_exists($uploadPath)) {
            mkdir($uploadPath, 0755, true);
        }

        for ($i = 1; $i <= 6; $i++) {
            $inputName = "project_image{$i}";
            if ($request->hasFile($inputName)) {
                $file = $request->file($inputName);
                $filename = "{$inputName}_" . time() . '.' . $file->getClientOriginalExtension();
                $file->move($uploadPath, $filename);
                $data[$inputName] = $filename;
            }
        }

        WebProject::create($data);

        return redirect()->route('webimage')->with('status', 'Project images uploaded successfully!');
    }
    public function webprojectedit()
    {
        $data = WebProject::first(); // Assuming only one set of project images
        return view('admin.web-project-edit', compact('data'));
    }
    public function webprojectupdate(Request $request)
    {
        $projectImage = WebProject::first();
        $uploadPath = public_path('uploads/project');

        if (!file_exists($uploadPath)) {
            mkdir($uploadPath, 0755, true);
        }

        $data = [];

        for ($i = 1; $i <= 6; $i++) {
            $inputName = "project_image{$i}";
            if ($request->hasFile($inputName)) {
                if (!empty($projectImage->$inputName)) {
                    $oldPath = public_path('uploads/project/' . $projectImage->$inputName);
                    if (file_exists($oldPath)) {
                        unlink($oldPath);
                    }
                }
                $file = $request->file($inputName);
                $filename = "{$inputName}_" . time() . '.' . $file->getClientOriginalExtension();
                $file->move($uploadPath, $filename);
                $data[$inputName] = $filename;
            }
        }

        $projectImage->update($data);

        return redirect()->route('webimage')->with('status', 'Project images updated successfully!');
    }
    public function webcontact()
    {
        return view('admin.web-contact');
    }

    public function webcontactedit()
    {
        $contact = WebContact::first(); // Assuming only one image
        return view('admin.web-contact-edit', compact('contact'));
    }

    public function webcontactStore(Request $request)
    {
        $uploadPath = public_path('uploads/contact');

        if (!file_exists($uploadPath)) {
            mkdir($uploadPath, 0755, true);
        }

        $data = [];

        if ($request->hasFile('contact_image')) {
            $file = $request->file('contact_image');
            $filename = 'contact_image_' . time() . '.' . $file->getClientOriginalExtension();
            $file->move($uploadPath, $filename);
            $data['contact_image'] = $filename;
        }

        WebContact::create($data);

        return redirect()->route('webimage')->with('status', 'Contact image uploaded successfully!');
    }


    public function webcontactUpdate(Request $request)
    {
        $contact = WebContact::first();
        $uploadPath = public_path('uploads/contact');

        if (!file_exists($uploadPath)) {
            mkdir($uploadPath, 0755, true);
        }

        if ($request->hasFile('contact_image')) {
            // Delete old image if exists
            if ($contact->contact_image && file_exists($uploadPath . '/' . $contact->contact_image)) {
                unlink($uploadPath . '/' . $contact->contact_image);
            }

            $file = $request->file('contact_image');
            $filename = 'contact_image_' . time() . '.' . $file->getClientOriginalExtension();
            $file->move($uploadPath, $filename);
            $contact->contact_image = $filename;
        }

        $contact->save();

        return redirect()->route('webimage')->with('status', 'Contact image updated successfully!');
    }
    public function webdonation()
    {
        return view('admin.web-donation');
    }
    public function webdonationstore(Request $request)
    {
        $uploadPath = public_path('uploads/donation');

        if (!file_exists($uploadPath)) {
            mkdir($uploadPath, 0755, true);
        }

        $data = [];

        if ($request->hasFile('donation_image')) {
            $file = $request->file('donation_image');
            $filename = 'donation_image_' . time() . '.' . $file->getClientOriginalExtension();
            $file->move($uploadPath, $filename);
            $data['donation_image'] = $filename;
        }

        WebDonation::create($data);

        return redirect()->route('webimage')->with('status', 'Donation image uploaded successfully!');
    }
    public function webdonationedit()
    {
        $donation = WebDonation::first();
        return view('admin.web-donation-edit', compact('donation'));
    }
    public function webdonationupdate(Request $request)
    {
        $donation = WebDonation::first();
        $uploadPath = public_path('uploads/donation');

        if (!file_exists($uploadPath)) {
            mkdir($uploadPath, 0755, true);
        }

        if ($request->hasFile('donation_image')) {
            // Delete old image if exists
            if ($donation->donation_image && file_exists($uploadPath . '/' . $donation->donation_image)) {
                unlink($uploadPath . '/' . $donation->donation_image);
            }

            $file = $request->file('donation_image');
            $filename = 'donation_image_' . time() . '.' . $file->getClientOriginalExtension();
            $file->move($uploadPath, $filename);
            $donation->donation_image = $filename;
        }

        $donation->save();

        return redirect()->route('webimage')->with('status', 'Donation image updated successfully!');
    }

    //Certificate Start Now

    public function certificate()
    {
        return view('admin.certificate');
    }
    public function certificateCreate()
    {
        return view('admin.certificate-create');
    }
    public function certificateStore(Request $request)
    {
        $request->validate([
            'title' => 'required|string',
            'certificate_image' => 'nullable|image',
            'description' => 'nullable|string',
            'is_feature' => 'required|string',
            'status' => 'required|string',
        ]);

        $data = $request->only('title', 'description', 'is_feature', 'status');

        if ($request->hasFile('certificate_image')) {
            $file = $request->file('certificate_image');
            $filename = 'certificate_' . time() . '.' . $file->getClientOriginalExtension();
            $file->move(public_path('uploads/certificates'), $filename);
            $data['certificate_image'] = $filename;
        }

        Certificate::create($data);

        return redirect()->route('certificate')->with('status', 'Certificate added successfully!');
    }


    public function getCertificateData()
    {
        $categorys = Certificate::orderBy('id', 'desc')->get();
        $result = [];
        $i = 1;

        foreach ($categorys as $category) {
            // Create status dropdown
            $statusOptions = '
                <select class="form-control certificate-status" data-id="' . $category->id . '">
                    <option value="Active"' . ($category->status === 'Active' ? ' selected' : '') . '>Active</option>
                    <option value="Inactive"' . ($category->status === 'Inactive' ? ' selected' : '') . '>Inactive</option>
                </select>';

            $editBtn = "<a class='btn btn-info editBtn' data-id='{$category->id}'>Edit</a>";
            $deleteBtn = "<a class='btn btn-danger deleteBtn' data-id='{$category->id}'>Delete</a>";

            $result['data'][] = [
                $i++,
                $category->title,
                $statusOptions,
                $editBtn . ' ' . $deleteBtn
            ];
        }

        return response()->json($result);
    }

    public function certificateEdit($id)
    {
        $certificate = Certificate::findOrFail($id);
        return view('admin.certificate-edit', compact('certificate'));
    }

    public function certificateUpdate(Request $request, $id)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'nullable|string',
            'status' => 'required|in:Active,Inactive',
        ]);

        $certificate = Certificate::findOrFail($id);
        $certificate->title = $request->title;
        $certificate->description = $request->description;
        $certificate->is_feature = $request->is_feature;
        $certificate->status = $request->status;

        if ($request->hasFile('certificate_image')) {
            if ($certificate->certificate_image && file_exists(public_path('uploads/certificates/' . $certificate->certificate_image))) {
                unlink(public_path('uploads/certificates/' . $certificate->certificate_image));
            }

            $file = $request->file('certificate_image');
            $filename = 'certificate_' . time() . '.' . $file->getClientOriginalExtension();
            $file->move(public_path('uploads/certificates'), $filename);

            $certificate->certificate_image = $filename;
        }

        $certificate->save();

        return redirect()->route('certificate')->with('status', 'Certificate updated successfully.');
    }


    public function certificateUpdateStatus(Request $request)
    {
        $category = Certificate::find($request->id);

        if (!$category) {
            return response()->json(['message' => 'Certificate not found'], 404);
        }

        $category->status = $request->status;
        $category->save();

        return response()->json(['message' => 'Status Updated Successfully']);
    }

    public function certificateDelete($id)
    {
        $category = Certificate::findOrFail($id);
        $category->delete();

        return response()->json(['status' => 'success', 'message' => 'Certificate deleted successfully.']);
    }


    //Web Settings Start Now

    public function websettings()
    {
        $hasWebSettings = \App\Models\Websettings::exists();
        return view('admin.websettings', compact('hasWebSettings'));
    }

    public function websettingsCreate()
    {
        return view('admin.websettings-create');
    }

    public function websettingsStore(Request $request)
    {
        $request->validate([
            'company_name' => 'required|string',
            'company_email' => 'required|string',
            'company_phone' => 'required|string',
            'websettings_image' => 'nullable|image',
            'company_address' => 'nullable|string',
            'status' => 'required|string',
        ]);

        $data = $request->only('company_name', 'company_email', 'company_phone', 'company_address', 'status');

        if ($request->hasFile('websettings_image')) {
            $file = $request->file('websettings_image');
            $filename = 'websettings_' . time() . '.' . $file->getClientOriginalExtension();
            $file->move(public_path('uploads/websettings'), $filename);
            $data['websettings_image'] = $filename;
        }

        Websettings::create($data);

        return redirect()->route('websettings')->with('status', 'Web Settings added successfully!');
    }

    public function getwebsettingsData()
    {
        $categorys = Websettings::orderBy('id', 'desc')->get();
        $result = [];
        $i = 1;

        foreach ($categorys as $category) {
            // Create status dropdown
            $statusOptions = '
                <select class="form-control websettings-status" data-id="' . $category->id . '">
                    <option value="Active"' . ($category->status === 'Active' ? ' selected' : '') . '>Active</option>
                    <option value="Inactive"' . ($category->status === 'Inactive' ? ' selected' : '') . '>Inactive</option>
                </select>';

            $editBtn = "<a class='btn btn-info editBtn' data-id='{$category->id}'>Edit</a>";
            $deleteBtn = "<a class='btn btn-danger deleteBtn' data-id='{$category->id}'>Delete</a>";

            $logoUrl = asset('uploads/websettings/' . $category->websettings_image); // change 'logo' if your DB column is different
            $imageTag = '<img src="' . $logoUrl . '" alt="Logo" height="50">';

            $result['data'][] = [
                $i++,
                $imageTag,
                $category->company_name,
                $statusOptions,
                $editBtn . ' ' . $deleteBtn
            ];
        }

        return response()->json($result);
    }

    public function websettingsUpdateStatus(Request $request)
    {
        $category = Websettings::find($request->id);

        if (!$category) {
            return response()->json(['message' => 'Web Settings not found'], 404);
        }

        $category->status = $request->status;
        $category->save();

        return response()->json(['message' => 'Status Updated Successfully']);
    }

    public function websettingsEdit($id)
    {
        $websettings = Websettings::findOrFail($id);
        return view('admin.websettings-edit', compact('websettings'));
    }

    public function websettingsUpdate(Request $request, $id)
    {
        $request->validate([
            'company_name' => 'required|string',
            'company_email' => 'required|email',
            'company_phone' => 'required|string',
            'company_address' => 'nullable|string',
            'facebook_link' => 'nullable|url',
            'insta_link' => 'nullable|url',
            'twitter_link' => 'nullable|url',
            'linkdin_link' => 'nullable|url',
            'status' => 'required|in:Active,Inactive',
        ]);

        $websettings = Websettings::findOrFail($id);

        $websettings->company_name = $request->company_name;
        $websettings->company_email = $request->company_email;
        $websettings->company_phone = $request->company_phone;
        $websettings->company_address = $request->company_address;
        $websettings->facebook_link = $request->facebook_link;
        $websettings->insta_link = $request->insta_link;
        $websettings->twitter_link = $request->twitter_link;
        $websettings->linkdin_link = $request->linkdin_link;
        $websettings->status = $request->status;

        if ($request->hasFile('websettings_image')) {
            if ($websettings->websettings_image && file_exists(public_path('uploads/websettings/' . $websettings->websettings_image))) {
                unlink(public_path('uploads/websettings/' . $websettings->websettings_image));
            }

            $file = $request->file('websettings_image');
            $filename = 'websettings' . time() . '.' . $file->getClientOriginalExtension();
            $file->move(public_path('uploads/websettings'), $filename);

            $websettings->websettings_image = $filename;
        }

        $websettings->save();

        return redirect()->route('websettings')->with('status', 'Web Settings Updated Successfully.');
    }


    public function websettingsDelete($id)
    {
        $category = Websettings::findOrFail($id);
        $category->delete();

        return response()->json(['status' => 'success', 'message' => 'Web Details deleted successfully.']);
    }

    //Users Start Now

    public function users()
    {
        return view('admin.users');
    }

    public function getusersData()
    {
        $donations = Donation::orderBy('id', 'desc')->get();
        $result = [];
        $i = 1;

        foreach ($donations as $donation) {

            $editBtn = "<a class='btn btn-info editBtn' data-id='{$donation->id}'>View</a>";
            $deleteBtn = "<a class='btn btn-danger deleteBtn' data-id='{$donation->id}'>Delete</a>";
            $formattedDate = Carbon::parse($donation->created_at)->format('d-m-Y h:i A');
            $formattedAmount = '₹' . number_format($donation->amount, 2);


            $result['data'][] = [
                $i++,
                $donation->first_name,
                $donation->email,
                $donation->phone,
                $formattedAmount,
                $formattedDate,
                $editBtn . ' ' . $deleteBtn
            ];
        }

        return response()->json($result);
    }

    public function usersEdit($id)
    {
        $donation = Donation::findOrFail($id);
        return view('admin.users-edit', compact('donation'));
    }

    public function usersDelete($id)
    {
        $category = Donation::findOrFail($id);
        $category->delete();

        return response()->json(['status' => 'success', 'message' => 'Users deleted successfully.']);
    }
    //users end

    //user contactus message and details Start
    public function contactMessageView()
    {
        return view('admin.contact-message');
    }

    public function contactMessageData()
    {
        $messages = ContactMessage::latest()->get();
        $result = [];
        $i = 1;

        foreach ($messages as $msg) {
            $formattedDate = Carbon::parse($msg->created_at)->format('d-m-Y h:i A');
            $result['data'][] = [
                $i++,
                $msg->name,
                $msg->email,
                $msg->phone ?? '—',
                $msg->message,
                $formattedDate// automatically decrypted
            ];
        }

        return response()->json($result);
    }



    //donation Listting
    public function donation()
    {
        return view('admin/donation');
    }

    public function donationData()
    {
        $donations = Donation::latest()->get();
        $result = [];
        $i = 1;

        foreach ($donations as $donation) {
            $formattedDate = Carbon::parse($donation->created_at)->format('d-m-Y h:i A');

            $printUrl = route('donation.print', $donation->id);

            $result[] = [
                'sl_no' => $i++,
                'name' => $donation->first_name,
                'email' => $donation->email,
                'phone' => $donation->phone,
                'amount' => '₹' . number_format($donation->amount, 2),
                'payment_date' => $formattedDate,
                'actions' => '
                <a href="' . $printUrl . '" target="_blank" class="btn btn-primary btn-sm">Print</a>
                <form action="' . route('donation.send', $donation->id) . '" method="POST" style="display:inline-block;">
                    ' . csrf_field() . '
                    <select name="send_via" onchange="this.form.submit()" class="form-select form-select-sm">
                        <option value="">Send via</option>
                        <option value="email">Email</option>
                        <option value="whatsapp">WhatsApp</option>
                    </select>
                </form>'
            ];
        }

        return response()->json(['data' => $result]);
    }
    public function printCertificate($id)
    {
        $donation = Donation::findOrFail($id);

        // Load view and pass donation data dynamically
        $pdf = Pdf::loadView('admin.certificate-layout', compact('donation'));

        // Stream PDF to browser
        return $pdf->stream('certificate_' . $donation->id . '.pdf');
    }
    public function sendCertificate(Request $request, $id)
    {
        $request->validate(['send_via' => 'required|in:email,whatsapp']);
        $donation = Donation::findOrFail($id);
        $sendVia = $request->send_via;

        // Generate PDF
        $pdf = Pdf::loadView('admin.certificate-layout', compact('donation'))->output();

        if ($sendVia === 'email') {
            // Send via Email
            Mail::send([], [], function ($message) use ($donation, $pdf) {
                $message->to($donation->email)
                    ->subject('Your Donation Certificate')
                    ->attachData($pdf, 'certificate.pdf', [
                        'mime' => 'application/pdf',
                    ])
                    ->text('Thank you for your donation. Please find your certificate attached.');
            });

            return back()->with('status', 'Certificate sent via Email.');
        }

        if ($sendVia === 'whatsapp') {



            $whatsappNumber = preg_replace('/[^0-9]/', '', $donation->phone); // clean number
            $messageText = urlencode("Thank you for your donation. Download your certificate here: $pdf");
            $waUrl = "https://wa.me/{$whatsappNumber}?text={$messageText}";

            return redirect($waUrl);
        }
    }
    public function frontvideo()
    {
        return view('admin.video');
    }
    public function getVideoData()
    {
        $videos = Video::orderBy('id', 'desc')->get();
        $result = [];
        $i = 1;

        foreach ($videos as $video) {
            // Status dropdown
            $statusOptions = '
            <select class="form-control video-status" data-id="' . $video->id . '">
                <option value="1"' . ($video->status == "Active" ? ' selected' : '') . '>Active</option>
                <option value="0"' . ($video->status == "Inactive" ? ' selected' : '') . '>Inactive</option>
            </select>';

            // Feature dropdown
            $featureOptions = '
            <select class="form-control video-feature" data-id="' . $video->id . '">
                <option value="1"' . ($video->is_feature == 1 ? ' selected' : '') . '>Yes</option>
                <option value="0"' . ($video->is_feature == 0 ? ' selected' : '') . '>No</option>
            </select>';

            // Action buttons
            $editBtn = "<a href='" . route('admin.video.edit', $video->id) . "' class='btn btn-sm btn-primary'>Edit</a>";
            $deleteBtn = "
            <form action='" . route('admin.video.delete', $video->id) . "' method='POST' style='display:inline;'>
                " . csrf_field() . method_field('DELETE') . "
                <button type='submit' onclick='return confirm(\"Are you sure?\");' class='btn btn-sm btn-danger'>Delete</button>
            </form>";

            $result['data'][] = [
                $i++,
                $video->video_link,
                $statusOptions,
                $featureOptions,
                $editBtn . ' ' . $deleteBtn
            ];
        }

        return response()->json($result);
    }

    public function createvideo()
    {
        return view('admin.video-create');
    }
    public function storeVideo(Request $request)
    {
        $request->validate([
            'video_link' => 'required|url',
            'is_feature' => 'required|boolean',
            'status' => 'required|in:Active,Inactive',
        ]);

        Video::create($request->all());
        return redirect()->route('admin.video')->with('status', 'Video created successfully.');
    }
    public function editVideo($id)
    {
        $video = Video::findOrFail($id);
        return view('admin.video-create', compact('video'));
    }

    public function updateVideo(Request $request, $id)
    {
        $request->validate([
            'video_link' => 'required|url',
            'is_feature' => 'required|boolean',
            'status' => 'required',
        ]);

        $video = Video::findOrFail($id);
        $video->update($request->all());

        return redirect()->route('admin.video')->with('status', 'Video updated successfully.');
    }

    public function destroyVideo($id)
    {
        $video = Video::findOrFail($id);
        $video->delete();

        return redirect()->route('admin.video')->with('status', 'Video deleted successfully.');
    }




}
